﻿using System;
using System.Collections.Generic;
using System.Data.Linq;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using VIRP.EFR.BLL;
using VIRP.EFR.BO;

namespace VIRP.EFR.DAL
{
    public class DoDFragmentDB : DBUtils
    {
        private EFRDB2DataContext efrContext;

        public DoDFragmentDB()
        {
        }

        private int selectFragmentsByPatientIdCount = 0;
        private int selectDoDFragmentDetailsCount = 0;
        private int selectDoDFragmentsRawDataByPatientIdCount = 0;

        /// <summary>
        /// Select DoD_FRAGMENT_DATA_RECEIVELOG by patient SSN
        /// </summary>
        /// <param name="ssn"></param>
        /// <returns></returns>
        public DoD_FRAGMENT_DATA_RECEIVELOG SelectBySSN(string ssn)
        {
            using (efrContext = GetEfr2DataContext())
            {
                DoD_FRAGMENT_DATA_RECEIVELOG fragment = SelectBySSNLinqFilter(ssn).FirstOrDefault();

                return fragment;
            }
        }

        private IQueryable<DoD_FRAGMENT_DATA_RECEIVELOG> SelectBySSNLinqFilter(string ssn)
        {
            IQueryable<DoD_FRAGMENT_DATA_RECEIVELOG> linqFilter = (from e in efrContext.DoD_FRAGMENT_DATA_RECEIVELOGs
                                                                   where e.PATIENTSSN.Replace(" ", "").Replace("-", "").Equals(ssn.Replace(" ", "").Replace("-", ""))
                                                                   select e);
            return linqFilter;
        }

        private void SetLoadWith(EFRDB2DataContext db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<DOD_FRAGMENT_LAB>(e => e.DOD_FRAGMENTs);
            lo.LoadWith<DOD_FRAGMENT>(e => e.DOD_FRAGMENT_DETAILs);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        private void SetLoadWithAnalytes(EFRDB2DataContext db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<DOD_FRAGMENT>(e => e.DOD_FRAGMENT_DETAILs);
            lo.LoadWith<DOD_FRAGMENT_DETAIL>(e => e.STD_FRAGMENT_ANALYSIS_TYPE);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        private void SetLoadWithFragment(EFRDB2DataContext db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<DOD_FRAGMENT>(e => e.DOD_FRAGMENT_DETAILs);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        private void SetLoadWithFragmentDetail(EFRDB2DataContext db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<DOD_FRAGMENT_DETAIL>(e => e.STD_FRAGMENT_ANALYSIS_TYPE);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        //private void SetLoadWithDoDRawData(EFRDBDataContext db)
        //{
        //    DataLoadOptions lo = new DataLoadOptions();
        //    lo.LoadWith<DoD_FRAGMENT_DATA_RECEIVELOG>(e => e.DOD_FRAGMENT_RAW_DATA);

        //    db.LoadOptions = lo;
        //    db.DeferredLoadingEnabled = false;
        //}

        /// <summary>
        /// Get the last extract file date for each DoD lab
        /// </summary>
        /// <returns></returns>
        public IEnumerable<DoDFragmentDataExtractByLab> GetLastExtractDateForAllLabs(string sortOrder, string orderBy, string filter)
        {
            IEnumerable<DoDFragmentDataExtractByLab> results = null;

            using (efrContext = GetEfr2DataContext())
            {
                results = efrContext.usp_EFR_GetDoDFragmentFiles(sortOrder, orderBy, filter).ToList();
            }

            return results;
        }

        /// <summary>
        /// Get the last DoD Fragment Data Extract Date
        /// </summary>
        /// <returns></returns>
        public System.Nullable<DateTime> GetLastExtractDate(string ssn)
        {
            using (efrContext = GetEfr2DataContext())
            {
                System.Nullable<DateTime> extractDate = null;
                int logCount = (from e in efrContext.DoD_FRAGMENT_DATA_RECEIVELOGs
                                where e.PATIENTSSN.Replace(" ", "").Replace("-", "").Equals(ssn.Replace(" ", "").Replace("-", ""))
                                select e).Count();
                if (logCount > 0)
                {
                    extractDate = (from e in efrContext.DoD_FRAGMENT_DATA_RECEIVELOGs
                                   where e.PATIENTSSN.Replace(" ", "").Replace("-", "").Equals(ssn.Replace(" ", "").Replace("-", ""))
                                   select e.ORIGINALINPUTDATE).Min();
                }

                return extractDate;
            }
        }

        /// <summary>
        /// Select DoD Fragment Raw Data by Patient ID
        /// </summary>
        /// <param name="patientId"></param>
        /// <param name="sort"></param>
        /// <param name="maxRows"></param>
        /// <param name="startRow"></param>
        /// <returns></returns>
        public IEnumerable<usp_EFR_GetDoDFragmentFilesByPatientIdResult> SelectDoDFragmentsRawDataByPatientId(int patientId, string sort, int maxRows, int startRow)
        {
            IEnumerable<usp_EFR_GetDoDFragmentFilesByPatientIdResult> results = null;

            if (patientId > 0)
            {
                //PatientManager patientManager = new PatientManager();
                //PATIENT patient = patientManager.SelectByPatientID(patientId);

                //if (patient != null)
                //{
                //    if (string.IsNullOrEmpty(sort))
                //    {
                //        sort = "DOD_FRAGMENT_RAW_DATA_ID desc";
                //    }

                using (efrContext = GetEfr2DataContext())
                {
                    //SetLoadWithDoDRawData(efrContext);
                    results = efrContext.usp_EFR_GetDoDFragmentFilesByPatientId(Convert.ToString(patientId)).OrderBy(sort).ToList();
                    selectDoDFragmentsRawDataByPatientIdCount = results.Count();
                }
            }
            return results;
        }

        /// <summary>
        /// Count for Select DoD Fragment Raw Data by Patient ID
        /// </summary>
        /// <param name="patientId"></param>
        /// <returns></returns>
        public int SelectDoDFragmentsRawDataByPatientIdCount(int patientId)
        {
            return selectDoDFragmentsRawDataByPatientIdCount;
        }

        /// <summary>
        /// Select a record from DOD_FRAGMENT_RAW_DATA table by Raw Data ID
        /// </summary>
        /// <param name="rawDataId"></param>
        /// <returns></returns>
        public DOD_FRAGMENT_RAW_DATA SelectByRawDataId(int rawDataId)
        {
            using (efrContext = GetEfr2DataContext())
            {
                DOD_FRAGMENT_RAW_DATA rawData = (from e in efrContext.DOD_FRAGMENT_RAW_DATAs
                                                 where e.DOD_FRAGMENT_RAW_DATA_ID == rawDataId
                                                 select e).FirstOrDefault();

                return rawData;
            }
        }

        /// <summary>
        /// Get all fragments for a patient
        /// </summary>
        /// <param name="patientId"></param>
        /// <param name="sort"></param>
        /// <param name="maxRows"></param>
        /// <param name="startRow"></param>
        /// <returns></returns>
        public IEnumerable<DOD_FRAGMENT_LAB> SelectFragmentsByPatientId(int patientId)
        {
            IEnumerable<DOD_FRAGMENT_LAB> results = null;

            if (patientId > 0)
            {
                using (efrContext = GetEfr2DataContext())
                {
                    results = SelectFragmentsByPatientIdLinqFilter(patientId).OrderBy("LAB_NAME").ToList();
                    selectFragmentsByPatientIdCount = results.Count();
                }
            }
            return results;
        }

        /// <summary>
        /// Select fragments by lab ID
        /// </summary>
        /// <param name="labId"></param>
        /// <returns></returns>
        public IEnumerable<DOD_FRAGMENT> SelectFragmentsByLabId(int labId)
        {
            IEnumerable<DOD_FRAGMENT> results = null;

            if (labId > 0)
            {
                using (efrContext = GetEfr2DataContext())
                {
                    results = SelectFragmentsByLabIdLinqFilter(labId).OrderBy("DOD_FRAGMENT_ID").ToList();
                }
            }
            return results;
        }

        /// <summary>
        /// Select fragment analytes by fragment ID
        /// </summary>
        /// <param name="fragmentId"></param>
        /// <returns></returns>
        public IEnumerable<DOD_FRAGMENT_DETAIL> SelectFragmentAnalytesByFragmentId(int fragmentId)
        {
            IEnumerable<DOD_FRAGMENT_DETAIL> results = null;

            if (fragmentId > 0)
            {
                using (efrContext = GetEfr2DataContext())
                {
                    SetLoadWithAnalytes(efrContext);
                    results = SelectFragmentAnalytesByFragmentIdLinqFilter(fragmentId).OrderBy("DOD_FRAGMENT_DETAIL_ID").ToList();
                }
            }
            return results;
        }

        private IQueryable<DOD_FRAGMENT_LAB> SelectFragmentsByPatientIdLinqFilter(int patientId)
        {
            return (from t in efrContext.DOD_FRAGMENT_LABs
                    where t.PATIENT_ID == patientId
                    select t);
        }

        private IQueryable<DOD_FRAGMENT> SelectFragmentsByLabIdLinqFilter(int labId)
        {
            return (from t in efrContext.DOD_FRAGMENTs
                    where t.DOD_FRAGMENT_LAB_ID == labId
                    select t);
        }

        private IQueryable<DOD_FRAGMENT_DETAIL> SelectFragmentAnalytesByFragmentIdLinqFilter(int fragmentId)
        {
            return (from t in efrContext.DOD_FRAGMENT_DETAILs
                    where t.DOD_FRAGMENT_ID == fragmentId
                    select t);
        }

        //private IQueryable<DoD_FRAGMENT_DATA_RECEIVELOG> SelectDoDFragmentsRawDataByPatientIdLinqFilter(string ssn)
        //{
        //    IQueryable<DoD_FRAGMENT_DATA_RECEIVELOG> fullSet = null;
        //    IQueryable<DoD_FRAGMENT_DATA_RECEIVELOG> groupedSet = null;

        //    //Select all records for current patient
        //    fullSet = from t in efrContext.DoD_FRAGMENT_DATA_RECEIVELOGs
        //              where t.PATIENTSSN.Replace(" ", "").Replace("-", "").Equals(ssn.Replace(" ", "").Replace("-", ""))
        //              select t;

        //    if (fullSet != null)
        //    {
        //        //Group by raw data file
        //        groupedSet = from n in fullSet
        //                     group n by n.DOD_FRAGMENT_RAW_DATA_ID into g
        //                     select g.OrderByDescending(t => t.LAB_Report_DATE).FirstOrDefault();
        //    }

        //    return groupedSet;
        //}

        /// <summary>
        /// Get the count all fragments for a patient
        /// </summary>
        /// <param name="patientId"></param>
        public int SelectFragmentsByPatientIdCount(int patientId)
        {
            return selectFragmentsByPatientIdCount;
        }

        /// <summary>
        /// Select DoD_Fragment_DETAIL by DoD Fragment ID
        /// </summary>
        /// <param name="dodFragmentId"></param>
        /// <param name="sort"></param>
        /// <param name="maxRows"></param>
        /// <param name="startRow"></param>
        /// <returns></returns>
        public IEnumerable<DOD_FRAGMENT_DETAIL> SelectDoDFragmentDetails(int dodFragmentId, string sort, int maxRows, int startRow)
        {
            IEnumerable<DOD_FRAGMENT_DETAIL> results = null;

            using (efrContext = GetEfr2DataContext())
            {
                if (string.IsNullOrEmpty(sort))
                {
                    sort = "DOD_FRAGMENT_DETAIL_ID desc";
                }
                if (maxRows == 0)
                {
                    maxRows = 9999;
                }

                results = SelectDoDFragmentDetailsLinqFilter(dodFragmentId).OrderBy(sort).Skip(startRow).Take(maxRows).ToList();
                selectDoDFragmentDetailsCount = results.Count();
            }

            return results;
        }

        private IQueryable<DOD_FRAGMENT_DETAIL> SelectDoDFragmentDetailsLinqFilter(int dodFragmentId)
        {
            var linqFilter = from t in efrContext.DOD_FRAGMENT_DETAILs
                             where t.DOD_FRAGMENT_ID == dodFragmentId
                             select t;

            return linqFilter;
        }

        public int SelectDoDFragmentDetailsCount(int dodFragmentId)
        {
            return selectDoDFragmentDetailsCount;
        }

        /// <summary>
        /// Get DoD Fragment record by DoD fragment ID
        /// </summary>
        /// <param name="dodFragmentId"></param>
        /// <returns></returns>
        public DOD_FRAGMENT GetDoDFragmentById(int dodFragmentId)
        {
            using (efrContext = GetEfr2DataContext())
            {
                SetLoadWithFragment(efrContext);

                return (from f in efrContext.DOD_FRAGMENTs
                        where f.DOD_FRAGMENT_ID == dodFragmentId
                        select f).FirstOrDefault();
            }
        }
        
        /// <summary>
        /// Get a DoD fragment analyte by analyte ID
        /// </summary>
        /// <param name="analyteId"></param>
        /// <returns></returns>
        public DOD_FRAGMENT_DETAIL GetFragmentAnalyteById(int analyteId)
        {
            using (efrContext = GetEfr2DataContext())
            {
                SetLoadWithFragmentDetail(efrContext);

                return (from f in efrContext.DOD_FRAGMENT_DETAILs
                        where f.DOD_FRAGMENT_DETAIL_ID == analyteId
                        select f).FirstOrDefault();
            }
        }
        
        /// <summary>
        /// Get DoD fragment lab by ID
        /// </summary>
        /// <param name="labId"></param>
        /// <returns></returns>
        public DOD_FRAGMENT_LAB GetDoDFragmentLabById(int labId)
        {
            using (efrContext = GetEfr2DataContext())
            {
                SetLoadWith(efrContext);
                return (from f in efrContext.DOD_FRAGMENT_LABs
                        where f.DOD_FRAGMENT_LAB_ID == labId
                        select f).FirstOrDefault();
            }
        }

        /// <summary>
        /// Update DOD_FRAGMENT_LAB record
        /// </summary>
        /// <param name="lab"></param>
        /// <returns></returns>
        //public DOD_FRAGMENT_LAB Update(DOD_FRAGMENT_LAB lab)
        //{
        //    using (efrContext = GetEfrDataContext())
        //    {
        //        efrContext.DeferredLoadingEnabled = false;

        //        //lab.SynchroniseWithDataContext(efrContext); //this line traverses all entities, attaching all of them as appropriate to the data context.
        //        //var labDb = efrContext.DOD_FRAGMENT_LABs.Where(x => x.DOD_FRAGMENT_LAB_ID == lab.DOD_FRAGMENT_LAB_ID).SingleOrDefault();
        //        //if (labDb == null)
        //        //{

        //        //    efrContext.DOD_FRAGMENT_LABs.InsertOnSubmit(lab);
        //        //}
        //        try
        //        {
        //            efrContext.SubmitChanges(ConflictMode.ContinueOnConflict);

        //        }
        //        catch (ChangeConflictException)
        //        {
        //            efrContext.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
        //        }

        //    }

        //    return lab;
        //}
        
        /// <summary>
        /// Update DOD_FRAGMENT record
        /// </summary>
        /// <param name="lab"></param>
        /// <returns></returns>
        //public DOD_FRAGMENT Update(DOD_FRAGMENT fragment)
        //{
        //    using (efrContext = GetEfrDataContext())
        //    {
        //        efrContext.DeferredLoadingEnabled = false;

        //        fragment.SynchroniseWithDataContext(efrContext); //this line traverses all entities, attaching all of them as appropriate to the data context.

        //        try
        //        {
        //            efrContext.SubmitChanges(ConflictMode.ContinueOnConflict);

        //        }
        //        catch (ChangeConflictException)
        //        {
        //            efrContext.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
        //        }

        //    }

        //    return fragment;
        //}

        /// <summary>
        /// Update DOD_FRAGMENT_DETAIL record
        /// </summary>
        /// <param name="fragmentAnalyte"></param>
        /// <returns></returns>
        //public DOD_FRAGMENT_DETAIL Update(DOD_FRAGMENT_DETAIL fragmentAnalyte)
        //{
        //    using (efrContext = GetEfrDataContext())
        //    {
        //        efrContext.DeferredLoadingEnabled = false;

        //        fragmentAnalyte.SynchroniseWithDataContext(efrContext); //this line traverses all entities, attaching all of them as appropriate to the data context.

        //        try
        //        {
        //            efrContext.SubmitChanges(ConflictMode.ContinueOnConflict);

        //        }
        //        catch (ChangeConflictException)
        //        {
        //            efrContext.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
        //        }

        //    }

        //    return fragmentAnalyte;
        //}
    }
}
